/**
 * Copyright vuja22 ( http://wonderfl.net/user/vuja22 )
 * MIT License ( http://www.opensource.org/licenses/mit-license.php )
 * Downloaded from: http://wonderfl.net/c/QqIc
 */

// forked from kaikoga's Mine Sweeper Blitz

package {
    
    /**
     * Mine Sweeper Blitz
     * 
     * Instructions:
     *  - [Click] to open grid
     *  - [Shift + click] to flag
     *  - Super automated game play
     *  - Opening grids will cause game to speed up
     *  - Opening mines will cause game to slow, and increments mine+ count
     *  - The mine+ count will be added to the number of mines of next stage
     *  - Open more grids and chain combo to score
     * 
     * マインスイーパーブリッツ
     * 
     * 遊び方：
     * - [クリック] でマスを開きます
     * - [Shift + クリック] で旗を立てます
     * - オート旗立てオートマス開きです
     * - マスをたくさん開けるとゲームスピードがどんどん速くなります
     * - 地雷マスを開けてしまうとゲームスピードが遅くなり、地雷＋が増加します
     * - 次のステージの開始時、地雷＋の数だけ地雷が増加します
     * - たくさんマスを開いたりたくさん連鎖すると高得点です
     * 
     * @author kaikoga
     */
    import com.bit101.components.PushButton;
    import flash.display.Sprite;
    import flash.events.ErrorEvent;
    import flash.events.Event;
    import flash.net.navigateToURL;
    import flash.net.SharedObject;
    import flash.net.URLRequest;
    import flash.utils.escapeMultiByte;
    import net.wonderfl.score.basic.BasicScoreForm;
    import net.wonderfl.score.basic.BasicScoreRecordViewer;
    
    [SWF(width = "465", height = "465", frameRate = "30", backgroundColor="0xffffff")]
    public class MineSweeperBlitz extends Sprite {
        
        private var startButton:PushButton;
        private var recordButton:PushButton;
        private var tweetButton:PushButton;
        private var gauge:TensionGauge;
        private var mineSweeper:MineSweeper;
        private var highScore:int = 0;
        private var message:String;
        
        public function MineSweeperBlitz() {
            super();
            this.gauge = new TensionGauge(0, 400);
            this.mineSweeper = new MineSweeper(7, 70, this.gauge);
            this.mineSweeper.addEventListener(Event.INIT, this.onMineSweeperInit);
            this.mineSweeper.addEventListener(Event.COMPLETE, this.onMineSweeperComplete);
            this.addChild(this.mineSweeper);
            
            this.startButton = new PushButton(this, 0, 0, "Start", this.onClickStart);
            this.tweetButton = new PushButton(this, 250, 0, "twitter", this.onClickTweet);
            this.tweetButton.visible = false;
            this.recordButton = new PushButton(this, 360, 0, "Records", this.onClickRecord);
            
            this.addChild(this.gauge);
        }
        
        private function onClickStart(event:Event):void {
            this.mineSweeper.reset(true);
        }
        
        private function onClickRecord(event:Event = null):void {
            this.mineSweeper.reset(true);
            new BasicScoreRecordViewer(this, 122.5, 112.5, "RANKING", 20);
        }
        
        private function onClickTweet(event:Event):void {
            var url:String = "http://twitter.com/home/?status=" + escapeMultiByte("[Mine Sweeper Blitz] ") + this.message + escapeMultiByte(" http://wonderfl.net/c/ #MineSweeperBlitz");
            navigateToURL(new URLRequest(url), "_blank");
        }
        
        private function onMineSweeperInit(event:Event):void {
            this.tweetButton.visible = false;
        }
        
        private var bsf:BasicScoreForm;
        private function onMineSweeperComplete(event:ErrorEvent):void {
            this.tweetButton.visible = true;
            this.message = event.text;
            if (this.highScore < this.mineSweeper.score) {
                this.bsf = new BasicScoreForm(this, 92.5, 152.5, this.mineSweeper.score, "REGISTER SCORE", this.onMineSweeperCompleteComplete);
                this.highScore = this.mineSweeper.score;
            }
        }
        
        private function onMineSweeperCompleteComplete(event:Boolean):void {
            if (this.bsf) {
                this.removeChild(this.bsf);
                this.bsf = null;
                this.onClickRecord(null);
            }
        }
        
    }
    
}

import com.bit101.components.IndicatorLight;
import com.bit101.components.Label;
import com.bit101.components.ProgressBar;
import flash.display.Bitmap;
import flash.display.BitmapData;
import flash.display.BlendMode;
import flash.display.PixelSnapping;
import flash.display.Sprite;
import flash.events.ErrorEvent;
import flash.events.Event;
import flash.events.MouseEvent;
import flash.geom.ColorTransform;
import flash.geom.Matrix;
import flash.geom.Point;
import flash.geom.Rectangle;
import flash.utils.Dictionary;
import org.libspark.betweenas3.BetweenAS3;
import org.libspark.betweenas3.tweens.ITween;

class TensionGauge extends Sprite {
    
    private var level0:ProgressBar;
    private var level1:ProgressBar;
    private var level2:ProgressBar;
    private var level3:ProgressBar;
    private var level4:ProgressBar;
    private var bar:ProgressBar;
    
    private var _value:int;
    private var _level:int;
    
    private static const VALUE_FRACTION:int = 256;
    
    public function get value():int {
        return this._value;
    }
    public function set value(value:int):void {
        while (value >= VALUE_FRACTION) {
            value -= VALUE_FRACTION;
            this.level++;
        }
        while (value < 0) {
            value += VALUE_FRACTION;
            this.level--;
        }
        this._value = value;
    }
    
    public function get level():int {
        return this._level;
    }
    public function set level(value:int):void {
        this._level = value < 0 ? 0 : value > 5 ? 5 : value;
        this.level0.value = value > 0 ? 100 : 0;
        this.level1.value = value > 1 ? 100 : 0;
        this.level2.value = value > 2 ? 100 : 0;
        this.level3.value = value > 3 ? 100 : 0;
        this.level4.value = value > 4 ? 100 : 0;
    }
    
    public function TensionGauge(x:Number, y:Number) {
        super();
        this.x = x;
        this.y = y;
        this.level0 = new ProgressBar(this, 0, 0);
        this.level0.width = 11;
        this.level0.maximum = 100;
        this.level1 = new ProgressBar(this, 13, 0);
        this.level1.width = 11;
        this.level0.maximum = 100;
        this.level2 = new ProgressBar(this, 26, 0);
        this.level2.width = 11;
        this.level0.maximum = 100;
        this.level3 = new ProgressBar(this, 39, 0);
        this.level3.width = 11;
        this.level0.maximum = 100;
        this.level4 = new ProgressBar(this, 52, 0);
        this.level4.width = 11;
        this.level0.maximum = 100;
        this.bar = new ProgressBar(this, 65, 0);
        this.bar.width = 400;
        this.bar.maximum = VALUE_FRACTION;
        this.addEventListener(Event.ENTER_FRAME, this.onEnterFrame);
    }
    
    private function onEnterFrame(event:Event):void {
        this.bar.value = (this.bar.value + this._value) / 2;
    }
    
}

class MineSweeper extends Sprite {
    
    private static const MINES_MIN:int = 50;
    private static const MINES_MAX:int = 150;
    
    private var scoreLabel:Label;
    private var comboLabel:Label;
    private var minesLabel:Label;
    private var flagsLabel:Label;
    private var messageLabel:Label;
    private var minesBar:ProgressBar;
    private var indicator:IndicatorLight;
    
    private var minesBarTweenY:ITween;
    private var minesBarTweenR:ITween;
    
    private var _score:int = 0;
    public function get score():int {
        return this._score;
    }
    public function set score(value:int):void {
        this._score = value;
        this.scoreLabel.text = "Score: " + value;
    }
    
    private var _combo:int = 0;
    public function get combo():int {
        return this._combo;
    }
    public function set combo(value:int):void {
        this._combo = value;
        this.comboLabel.text = "Combo: " + value;
    }
    
    private var _minesToAdd:int = 0;
    public function get minesToAdd():int {
        return this._minesToAdd;
    }
    public function set minesToAdd(value:int):void {
        if (this._mines + value > MINES_MAX) {
            value = MINES_MAX - this._mines;
        }
        this._minesToAdd = value;
        this.refreshMinesView();
    }
    
    private var _mines:int = 0;
    public function get mines():int {
        return this._mines;
    }
    public function set mines(value:int):void {
        this._mines = value;
        this.refreshMinesView();
    }
    
    private var nowTween:ITween = null;
    private function refreshMinesView():void {
        var minesTotal:int = this._mines + this._minesToAdd;
        this.minesLabel.text = "Mines: " + this._mines + (this._minesToAdd ? ("+" + this._minesToAdd) : "");
        this.minesBar.value = minesTotal - MINES_MIN;
        var newTween:ITween = minesTotal > 125 ? this.minesBarTweenR : minesTotal > 100 ? this.minesBarTweenY : null;
        if (!this.active) {
            newTween = null;
        }
        if (this.nowTween != newTween) {
            if (this.nowTween) {
                this.nowTween.stop();
            }
            this.nowTween = newTween;
            if (this.nowTween) {
                this.nowTween.play();
            }
        }
    }
    
    private var _flags:int = 0;
    public function get flags():int {
        return this._flags;
    }
    public function set flags(value:int):void {
        this._flags = value;
        this.flagsLabel.text = "Flags: " + value;
    }
    
    private var messageOriginTime:Number;
    public function message(message:String):void {
        this.messageOriginTime = new Date().getTime();
        this.messageLabel.x = 470;
        this.messageLabel.text = message;
    }
    
    private var _active:Boolean = false;
    public function get active():Boolean {
        return this._active;
    }
    public function set active(value:Boolean):void {
        this._active = value;
        this.indicator.color = value ? 0x33ff33 : 0xff6633;
        this.indicator.isLit = true;
    }
    
    private var _paused:Boolean = false;
    public function get paused():Boolean {
        return this._paused;
    }
    public function set paused(value:Boolean):void {
        if (this.paused == value) {
            return;
        }
        this._paused = value;
        this.indicator.flash(value ? 0 : 240);
        if (value) {
            this.indicator.isLit = true;
        }
    }
    
    private var field:Vector.<Vector.<Cell>>;
    private var allCells:Vector.<Cell>;
    private var spareMineCell:Cell;
    
    private var openPendingCells:Dictionary;
    private var clearPendingCells:Dictionary;
    
    private var FIELD_WIDTH:Number = 30;
    private var FIELD_HEIGHT:Number = 20;
    
    private var gauge:TensionGauge;
    private var background:Background;
    
    private var waitFrame:int = 0;
    private var mineFrame:int = 0;
    private var stageCount:int = 0;
    public var coveredCellCount:int = 0;
    
    public static const COLOR_TRANSFORM_PENDING:ColorTransform = new ColorTransform(1.5, 1.5, 1.5, 1);
    public static const COLOR_TRANSFORM_CLEARING:ColorTransform = new ColorTransform(1.1, 1.1, 1.1, 1);
    public static const COLOR_TRANSFORM_INACTIVE:ColorTransform = new ColorTransform();
    
    public function getCell(x:int, y:int):Cell {
        if (x < 0 || x >= FIELD_WIDTH || y < 0 || y >= FIELD_HEIGHT) {
            return null;
        }
        return field[x][y];
    }
    
    public function getCellContent(x:int, y:int, actual:Boolean = true):String {
        var cell:Cell = this.getCell(x, y);
        if (actual) {
            if (!cell) {
                return "";
            } else if (cell.hasMine) {
                return "*";
            }
        } else {
            if (!cell) {
                return "";
            } else {
                switch (cell.state) {
                    case "F":
                    return "F";
                    case "*":
                    return "*";
                    case " ":
                    return " ";
                }
            }
        }
        return "0";
    }
    
    public function requestOpen(cell:Cell):void {
        if (!this.openPendingCells) {
            this.openPendingCells = new Dictionary();
        }
        if (this.paused && cell.hasMine) {
            cell.hasMine = false;
            this.spareMineCell.hasMine = true;
        }
        this.openPendingCells[cell] = cell;
        cell.transform.colorTransform = COLOR_TRANSFORM_PENDING;
    }
    
    public function requestClear(cell:Cell):void {
        if (!this.clearPendingCells) {
            this.clearPendingCells = new Dictionary();
        }
        for (var i:int = -1; i <= 1; i++) {
            for (var j:int = -1; j <= 1; j++) {
                var c:Cell = this.getCell(cell.gridX + i, cell.gridY + j);
                if (c && !c.cleared) {
                    this.clearPendingCells[c] = c;
                    c.transform.colorTransform = COLOR_TRANSFORM_CLEARING;
                }
            }
        }
    }
    
    public function takeDamage():void {
        this.minesToAdd++;
        this.gauge.level--;
        this.combo = 0;
    }
    
    public function takeScore():void {
        this.gauge.value++;
    }
    
    private function compareCells(cell1:Cell, cell2:Cell):int {
        return Math.random() > 0.5 ? 1 : -1;
    }
    
    public function reset(fullReset:Boolean = false):void {
        if (fullReset) {
            this.clearPendingCells = null;
            this.openPendingCells = null;
            this.score = 0;
            this.mines = MINES_MIN;
            this.active = true;
            this.dispatchEvent(new Event(Event.INIT));
            this.active = true;
            this.stageCount = 1;
            this.gauge.level = 2;
            this.gauge.value = 0;
        } else {
            this.mines += this.minesToAdd;
            this.active = this.mines < MINES_MAX;
            this.stageCount++;
        }
        this.combo = 0;
        this.minesToAdd = 0;
        if (this.active) {
            this.allCells.sort(this.compareCells);
            var i:int = 0;
            for each (var cell:Cell in this.allCells) {
                cell.reset(i++ < this.mines);
                this.spareMineCell = cell;
            }
            this.message("Mine Sweeper Blitz Stage " + this.stageCount + "  LETS GO GO GO");
        } else {
            var message:String = "Stages: " + (this.stageCount - 1) + "  Score: " + this.score;
            this.dispatchEvent(new ErrorEvent(Event.COMPLETE, false, false, message));
            this.mines = MINES_MAX;
            this.message("Stages Complete:  " + message);
        }
        this.paused = true;
    }
    
    public function MineSweeper(x:Number = 0, y:Number = 0, gauge:TensionGauge = null) {
        super();
        this.background = new Background();
        this.background.x = -15;
        this.background.y = -75;
        this.addChild(this.background);
        this.indicator = new IndicatorLight(this, 48, -28, 0xff0000);
        var labelBack:Sprite = new Sprite();
        labelBack.graphics.beginFill(0xffffff, 0.8);
        labelBack.graphics.drawRect(-7, 307, 480, 16)
        labelBack.graphics.drawRect(100, -45, 200, 30)
        this.addChild(labelBack);
        this.scoreLabel = new Label(this, 100, -45, "Score: 0");
        this.comboLabel = new Label(this, 100, -30, "Combo: 0");
        this.minesLabel = new Label(this, 200, -45, "Mines: 0");
        this.flagsLabel = new Label(this, 200, -30, "Flags: 0");
        this.messageLabel = new Label(this, 200, 305, "");
        this.messageLabel.width = 200;
        this.minesBar = new ProgressBar(this, 300, -30);
        this.minesBar.maximum = MINES_MAX - MINES_MIN;
        this.minesBar.width = 150;
        this.minesBarTweenY = BetweenAS3.tween(this.minesBar,
                { transform: { colorTransform: { blueOffset:-127 }}},
                { transform: { colorTransform: { blueOffset:0 }}},
                0.8
                );
        this.minesBarTweenY = BetweenAS3.serial(this.minesBarTweenY, BetweenAS3.reverse(this.minesBarTweenY));
        this.minesBarTweenY = BetweenAS3.repeat(this.minesBarTweenY, uint.MAX_VALUE);
        this.minesBarTweenR = BetweenAS3.tween(this.minesBar,
                { transform: { colorTransform: { greenOffset:-127, blueOffset:-127 }}},
                { transform: { colorTransform: { greenOffset:0, blueOffset:0 }}},
                0.5
                );
        this.minesBarTweenR = BetweenAS3.serial(this.minesBarTweenR, BetweenAS3.reverse(this.minesBarTweenR));
        this.minesBarTweenR = BetweenAS3.repeat(this.minesBarTweenR, uint.MAX_VALUE);
        this.x = x;
        this.y = y;
        this.gauge = gauge;
        this.gauge.level = 2;
        this.field = new Vector.<Vector.<Cell>>();
        this.allCells = new Vector.<Cell>();
        for (var i:int = 0; i < FIELD_WIDTH; i++) {
            var line:Vector.<Cell> = new Vector.<Cell>;
            this.field[i] = line;
            for (var j:int = 0; j < FIELD_HEIGHT; j++) {
                var cell:Cell = new Cell(this, i, j);
                line[j] = cell;
                this.allCells.push(cell);
                this.addChild(cell);
            }
        }
        this.coveredCellCount = FIELD_WIDTH * FIELD_HEIGHT;
        this.reset(true);
        this.addEventListener(Event.ENTER_FRAME, this.onEnterFrame);
    }
    
    private function onEnterFrame(event:Event):void {
        this.messageLabel.x = 470 - (new Date().getTime() - this.messageOriginTime) / 15;
        if (!this.active) {
            this.background.frameMove();
            if (++this.waitFrame >= 52 - (this.gauge.level << 3)) {
                this.waitFrame = 0;
                this.background.excite(this.gauge.level);
            }
            return;
        }
        this.background.frameMove();
        if (++this.waitFrame >= 26 - (this.gauge.level << 2)) {
            this.waitFrame = 0;
            this.background.excite(this.gauge.level);
            var cells:Dictionary;
            var cell:Cell;
            if (this.coveredCellCount == 0 && this._mines == this._flags) {
                this.reset();
                return;
            }
            if (this.clearPendingCells) {
                cells = this.clearPendingCells;
                this.clearPendingCells = null;
                for each (cell in cells) {
                    cell.transform.colorTransform = COLOR_TRANSFORM_INACTIVE;
                    cell.tryClear();
                }
                this.score += this.combo++;
            } else {
                this.combo = 0;
            }
            if (this.openPendingCells) {
                cells = this.openPendingCells;
                this.openPendingCells = null;
                for each (cell in cells) {
                    cell.transform.colorTransform = COLOR_TRANSFORM_INACTIVE;
                    cell.open();
                    this.score++;
                }
                this.paused = false;
            }
        }
        if (!this.paused) {
            if (++this.mineFrame >= 60) {
                this.mineFrame = 0;
                this.minesToAdd++;
            }
        }
    }
    
}

class Cell extends Sprite {
    
    private static const WIDTH:Number = 15;
    private static const HEIGHT:Number = 15;
    private var manager:MineSweeper;
    private var content:Sprite;
    private var bitmap:Bitmap;
    
    public var gridX:int;
    public var gridY:int;
    
    private var _frame:int = 0;
    
    public var hasMine:Boolean = false;
    public var cleared:Boolean = false;
    
    private var _state:String = " ";
    public function get state():String {
        return this._state;
    }
    public function set state(value:String):void {
        switch (this.state) {
            case "F":
            case "*":
            this.manager.flags--;
            break;
            case " ":
            this.manager.coveredCellCount--;
            break;
        }
        switch (value) {
            case "F":
            case "*":
            this.manager.flags++;
            break;
            case " ":
            this.manager.coveredCellCount++;
            break;
        }
        this.bitmap.bitmapData = CellBitmaps.instance.getCellBitmapData(value);
        this._state = value;
    }
    
    private var _nextState:String = " ";
    public function set nextState(value:String):void {
        this._nextState = value;
        if (this._frame == 0) {
            this.addEventListener(Event.ENTER_FRAME, this.onEnterFrame);
        }
        this._frame = 6;
    }
    
    private function doOpen(force:Boolean = false):void {
        if (this._state != " " && !force) {
            return;
        } else if (this.hasMine) {
            this.nextState = "*";
        } else {
            var mineCount:int = 0;
            for (var i:int = -1; i <= 1; i++) {
                for (var j:int = -1; j <= 1; j++) {
                    switch (this.manager.getCellContent(this.gridX + i, this.gridY + j)) {
                        case "*":
                        mineCount++;
                        break;
                    }
                }
            }
            this.nextState = String(mineCount);
        }
    }
    
    public function open():void {
        this.doOpen();
        this.manager.requestClear(this);
    }
    
    public function flag(toggles:Boolean):void {
        switch (this.state) {
            case " ":
            this.state = "F";
            this.manager.requestClear(this);
            break;
            case "F":
            if (toggles) {
                this.state = " ";
                this.manager.requestClear(this);
            }
            break;
        }
    }
    
    public function clear(withFlags:Boolean = false, force:Boolean = false):void {
        for (var i:int = -1; i <= 1; i++) {
            for (var j:int = -1; j <= 1; j++) {
                var cell:Cell = this.manager.getCell(this.gridX + i, this.gridY + j);
                if (cell && (cell.state == " " || force)) {
                    if (withFlags) {
                        cell.flag(false);
                    } else if (force) {
                        if (cell.hasMine) {
                            this.manager.minesToAdd++;
                            cell.state = "*";
                        } else if (cell.state == " " || cell.state == "F") {
                            cell.doOpen(true);
                        }
                    } else {
                        this.manager.requestOpen(cell);
                    }
                }
            }
        }
        this.cleared = true;
    }
    
    public function tryClear():void {
        switch (this._state) {
            case "B":
            case "F":
            case "*":
            this.cleared = true;
            break;
            case " ":
            break;
            default:
            var mineCount:int = int(this._state);
            var flagCount:int = 0;
            var coveredCount:int = 0;
            for (var i:int = -1; i <= 1; i++) {
                for (var j:int = -1; j <= 1; j++) {
                    switch (this.manager.getCellContent(this.gridX + i, this.gridY + j, false)) {
                        case "F":
                        case "*":
                        flagCount++;
                        break;
                        case " ":
                        coveredCount++;
                        break;
                    }
                }
            }
            if (mineCount == flagCount) {
                this.clear();
            } else if (mineCount == coveredCount + flagCount) {
                this.clear(true);
            }
            break;
        }
    }
    
    public function reset(mine:Boolean):void {
        this.nextState = " ";
        this.hasMine = mine;
        this.cleared = false;
    }
    
    public function Cell(manager:MineSweeper, gridX:int = 0, gridY:int = 0) {
        super();
        this.manager = manager;
        this.gridX = gridX;
        this.gridY = gridY;
        this.x = gridX * WIDTH;
        this.y = gridY * HEIGHT;
        this.content = new Sprite();
        this.content.x = WIDTH * 0.5;
        this.content.y = HEIGHT * 0.5;
        this.addChild(this.content);
        this.bitmap = new Bitmap(null);
        this.bitmap.x = WIDTH * -0.5;
        this.bitmap.y = HEIGHT * -0.5;
        this.state = " ";
        this.content.addChild(this.bitmap);
        this.content.rotationY = 0;
        this.addEventListener(MouseEvent.CLICK, this.onCliek);
    }
    
    private function onCliek(event:MouseEvent):void {
        if (this.manager.active) {
            if (event.shiftKey) {
                this.flag(true);
            } else {
                this.manager.requestOpen(this);
            }
        }
    }
    
    private function onEnterFrame(event:Event):void {
        switch (--this._frame) {
            case 5:
            case 4:
            this.content.rotationY = (6 - this._frame) * 30;
            break;
            case 3:
            this.state = this._nextState;
            this.content.rotationY = 90;
            break;
            case 2:
            case 1:
            this.content.rotationY = this._frame * 30;
            break;
            case 0:
            this.removeEventListener(Event.ENTER_FRAME, this.onEnterFrame);
            this.content.rotationY = 0;
            switch (this.state) {
                case "*":
                this.manager.takeDamage();
                this.clear(false, true);
                break;
                case " ":
                case "F":
                break;
                default:
                this.manager.takeScore();
                break;
            }
            break;
        }
    }
    
    override public function toString():String {
        return "[Cell (" + this.gridX + "," + this.gridY + "):" + this._state + "]";
    }
    
}

class CellBitmaps {
    
    public static var instance:CellBitmaps = new CellBitmaps();
    
    private var _bitmaps:Object = { };
    
    private function createBitmapData(colors:Object, data:Array):BitmapData {
        var result:BitmapData = new BitmapData(15, 15, false);
        for (var y:int = 0; y < 15; y++) {
            var line:String = data[y];
            for (var x:int = 0; x < 15; x++) {
                result.setPixel(x, y, colors[line.charAt(x)]);
            }
        }
        return result;
    }
    
    public function getCellBitmapData(char:String):BitmapData {
        var result:BitmapData = this._bitmaps[char];
        if (!result) {
            var colors:Object = {
                " ":0xcccccc,
                "A":0x333333,
                "B":0x666666,
                "C":0x999999,
                "1":0x003399,
                "2":0x339900,
                "3":0x993300,
                "4":0x003366,
                "5":0x660000,
                "6":0x009999,
                "7":0x333333,
                "8":0x666666,
                "*":0xFF0000
            }
            switch (char) {
                case " ":
                result = createBitmapData(colors, [
                        " BBBBBBBBBBBBBC",
                        "B BBBBBBBBBBBCA",
                        "BB           AA",
                        "BB           AA",
                        "BB           AA",
                        "BB           AA",
                        "BB           AA",
                        "BB           AA",
                        "BB           AA",
                        "BB           AA",
                        "BB           AA",
                        "BB           AA",
                        "BB           AA",
                        "BCAAAAAAAAAAABA",
                        "CAAAAAAAAAAAAAB"
                        ]);
                break;
                case "0":
                result = createBitmapData(colors, [
                        "CCCCCCCCCCCCCCC",
                        "C             C",
                        "C             C",
                        "C             C",
                        "C             C",
                        "C             C",
                        "C             C",
                        "C             C",
                        "C             C",
                        "C             C",
                        "C             C",
                        "C             C",
                        "C             C",
                        "C             C",
                        "CCCCCCCCCCCCCCC"
                        ]);
                break;
                case "1":
                result = createBitmapData(colors, [
                        "CCCCCCCCCCCCCCC",
                        "C             C",
                        "C      11     C",
                        "C     111     C",
                        "C    1111     C",
                        "C   1 111     C",
                        "C     111     C",
                        "C     111     C",
                        "C     111     C",
                        "C     111     C",
                        "C     111     C",
                        "C     111     C",
                        "C  111111111  C",
                        "C             C",
                        "CCCCCCCCCCCCCCC"
                        ]);
                break;
                case "2":
                result = createBitmapData(colors, [
                        "CCCCCCCCCCCCCCC",
                        "C             C",
                        "C    22222    C",
                        "C   222 222   C",
                        "C  222   222  C",
                        "C        222  C",
                        "C       222   C",
                        "C      222    C",
                        "C     222     C",
                        "C    222      C",
                        "C   222       C",
                        "C  222        C",
                        "C  222222222  C",
                        "C             C",
                        "CCCCCCCCCCCCCCC"
                        ]);
                break;
                case "3":
                result = createBitmapData(colors, [
                        "CCCCCCCCCCCCCCC",
                        "C             C",
                        "C    33333    C",
                        "C   333 333   C",
                        "C  333   333  C",
                        "C        333  C",
                        "C       333   C",
                        "C     3333    C",
                        "C       333   C",
                        "C        333  C",
                        "C  333   333  C",
                        "C   333 333   C",
                        "C    33333    C",
                        "C             C",
                        "CCCCCCCCCCCCCCC"
                        ]);
                break;
                case "4":
                result = createBitmapData(colors, [
                        "CCCCCCCCCCCCCCC",
                        "C             C",
                        "C        44   C",
                        "C       444   C",
                        "C      4444   C",
                        "C     44444   C",
                        "C    444444   C",
                        "C   444 444   C",
                        "C  444  444   C",
                        "C  444444444  C",
                        "C       444   C",
                        "C       444   C",
                        "C       444   C",
                        "C             C",
                        "CCCCCCCCCCCCCCC"
                        ]);
                break;
                case "5":
                result = createBitmapData(colors, [
                        "CCCCCCCCCCCCCCC",
                        "C             C",
                        "C  555555555  C",
                        "C  555        C",
                        "C  555        C",
                        "C  555        C",
                        "C  5555555    C",
                        "C       555   C",
                        "C        555  C",
                        "C        555  C",
                        "C  555   555  C",
                        "C   555 555   C",
                        "C    55555    C",
                        "C             C",
                        "CCCCCCCCCCCCCCC"
                        ]);
                break;
                case "6":
                result = createBitmapData(colors, [
                        "CCCCCCCCCCCCCCC",
                        "C             C",
                        "C       666   C",
                        "C      666    C",
                        "C     666     C",
                        "C    666      C",
                        "C   666       C",
                        "C  6666666    C",
                        "C  666  666   C",
                        "C  666   666  C",
                        "C  666   666  C",
                        "C   666 666   C",
                        "C    66666    C",
                        "C             C",
                        "CCCCCCCCCCCCCCC"
                        ]);
                break;
                case "7":
                result = createBitmapData(colors, [
                        "CCCCCCCCCCCCCCC",
                        "C             C",
                        "C  777777777  C",
                        "C  777   777  C",
                        "C  777   777  C",
                        "C        777  C",
                        "C       777   C",
                        "C      777    C",
                        "C     777     C",
                        "C     777     C",
                        "C     777     C",
                        "C     777     C",
                        "C     777     C",
                        "C             C",
                        "CCCCCCCCCCCCCCC"
                        ]);
                break;
                case "8":
                result = createBitmapData(colors, [
                        "CCCCCCCCCCCCCCC",
                        "C             C",
                        "C    88888    C",
                        "C   888 888   C",
                        "C  888   888  C",
                        "C  888   888  C",
                        "C   888 888   C",
                        "C    88888    C",
                        "C   888 888   C",
                        "C  888   888  C",
                        "C  888   888  C",
                        "C   888 888   C",
                        "C    88888    C",
                        "C             C",
                        "CCCCCCCCCCCCCCC"
                        ]);
                break;
                case "*":
                result = createBitmapData(colors, [
                        "CCCCCCCCCCCCCCC",
                        "C             C",
                        "C             C",
                        "C      *      C",
                        "C   *  *  *   C",
                        "C    *****    C",
                        "C    *****    C",
                        "C  *********  C",
                        "C    *****    C",
                        "C    *****    C",
                        "C   *  *  *   C",
                        "C      *      C",
                        "C             C",
                        "C             C",
                        "CCCCCCCCCCCCCCC"
                        ]);
                break;
                case "F":
                result = createBitmapData(colors, [
                        "CCCCCCCCCCCCCCC",
                        "C             C",
                        "C      *      C",
                        "C     **      C",
                        "C    ***      C",
                        "C   ****      C",
                        "C  *****      C",
                        "C      *      C",
                        "C      *      C",
                        "C      *      C",
                        "C    AAAAA    C",
                        "C  AAAAAAAAA  C",
                        "C  AAAAAAAAA  C",
                        "C             C",
                        "CCCCCCCCCCCCCCC"
                        ]);
                break;
            }
            this._bitmaps[char] = result;
        }
        return result;
    }
    
    public function CellBitmaps() {
        super();
    }
    
}

class Background extends Bitmap {
    
    private var nowSeed:int;
    private var nowLevel:int;
    private var workBitmapData:BitmapData;
    private var workRect:Rectangle= new Rectangle(0, 0, 32, 32);
    private var workRectLine:Rectangle= new Rectangle(0, 0, 32, 1);
    private var workColorTransform:ColorTransform = new ColorTransform(1, 1, 1, 0.125);
    private var workColorTransform1:ColorTransform = new ColorTransform(1, 1, 1, 1, 20, 20, 20, 0);
    private var workColorTransform3:ColorTransform = new ColorTransform(2, 2, 2, 1, -64, -64, -64, 0);
    private var workColorTransform4:ColorTransform = new ColorTransform(2.5, 2.5, 2.5, 1, -127, -127, -127, 0);
    
    private var perlinOffsets:Array = [new Point(), new Point(), new Point(), new Point()];
    public function Background() {
        super(new BitmapData(32, 32, false, 0xffffff), PixelSnapping.NEVER, false);
        this.scaleX = 15;
        this.scaleY = 15;
        this.workBitmapData = new BitmapData(32, 32, false, 0xffffff);
    }
    
    public function excite(level:int):void {
        this.nowLevel = level;
        switch (level) {
            case 0:
            this.nowSeed++;
            this.bitmapData.noise(nowSeed, 230, 240, 7, true);
            break;
            case 1:
            this.nowSeed++;
            this.workBitmapData.noise(nowSeed, 200, 220, 7, true);
            this.bitmapData.colorTransform(this.workRect, this.workColorTransform1)
            break;
            case 2:
            this.nowSeed++;
            this.workBitmapData.noise(nowSeed, 200, 255, 7, true);
            break;
            case 3:
            this.nowSeed++;
            this.workBitmapData.perlinNoise(32, 32, 4, nowSeed, true, false, 7, true, this.perlinOffsets);
            this.workBitmapData.colorTransform(this.workRect, this.workColorTransform3)
            break;
            case 4:
            this.workBitmapData.perlinNoise(32, 32, 4, nowSeed, true, false, 7, true, this.perlinOffsets);
            this.workBitmapData.colorTransform(this.workRect, this.workColorTransform4)
            Point(this.perlinOffsets[0]).y += 2;
            Point(this.perlinOffsets[1]).y += 1.4;
            Point(this.perlinOffsets[2]).y += 0.9;
            Point(this.perlinOffsets[3]).y += 0.2;
            break;
            case 5:
            this.nowSeed++;
            for (var y:int = 0; y < 32; y++) {
                this.workRectLine.y = y;
                switch ((y + this.nowSeed) & 3) {
                    case 0:
                    this.workBitmapData.fillRect(this.workRectLine, 0xffffff);
                    break;
                    case 1:
                    this.workBitmapData.fillRect(this.workRectLine, 0x999999);
                    break;
                    case 2:
                    this.workBitmapData.fillRect(this.workRectLine, 0x666666);
                    break;
                    case 3:
                    this.workBitmapData.fillRect(this.workRectLine, 0xcccccc);
                    break;
                }
            }
            break;
        }
    }
    
    public function frameMove():void {
        switch (this.nowLevel) {
            case 0:
            break;
            case 1:
            case 2:
            case 3:
            case 4:
            case 5:
            this.bitmapData.draw(this.workBitmapData, null, workColorTransform);
            break;
        }
    }
    
}
