/**
 * Copyright nitoyon ( http://wonderfl.net/user/nitoyon )
 * MIT License ( http://www.opensource.org/licenses/mit-license.php )
 * Downloaded from: http://wonderfl.net/c/7mZI
 */

package{
import flash.display.*;
import flash.events.Event;
import flash.utils.ByteArray;
import flash.geom.*;
import mx.utils.Base64Decoder;

[SWF(frameRate="10")]
public class Map3 extends Sprite{
	private const F:Number = 800;
	private const WIDTH:int = 240;
	private const HEIGHT:int = 300;
	private const FRAME:int = 64;
	private var ELEVATION:Number = 30;
	private var COLOR_SEA:uint = 0x7ebeea;
	private var COLORS:Array = [0x149614, 0xc8c800, 0x874700, 0x532f06, 0x675400, 0xffffff];

	public function Map3(){
		stage.align = "TL";
		stage.scaleMode = "noScale";
		var hw:Number = WIDTH / 2;
		var hh:Number = HEIGHT / 2;

		var decoder:Base64Decoder = new Base64Decoder();
		decoder.decode(E100N40_DATA);
		var bytes:ByteArray = decoder.toByteArray();
		bytes.uncompress();

		var bmdLine:BitmapData = createBmdLine();

		// 座標を MapPoint の配列で計算しておく
		var points:Array = [];
		for(var y:int = 0; y < HEIGHT; y++){
			for(var x:int = 0; x < WIDTH; x++){
				points.push(new MapPoint(x - hw, y - hh, bytes.readShort()));
			}
		}

		// 回転行列作っとく
		var matrix:Matrix = new Matrix();
		matrix.rotate(2 * Math.PI / FRAME);

		// Bitmap を作成
		var bmp:Bitmap = new Bitmap();
		bmp.scaleX = bmp.scaleY = 2;
		addChild(bmp);

		// BitmapData のキャッシュ
		var bmdCache:Array = [];

		// 毎フレーム処理
		var counter:int = 0;
		var rt:Rectangle = new Rectangle(0, 0, 1, ELEVATION * 2);
		var pt:Point = new Point();

		addEventListener("enterFrame", function(event:Event):void{
			// キャッシュがなければ作成する
			if (!bmdCache[counter]){
				var bmd:BitmapData = new BitmapData(WIDTH, HEIGHT * .6, false, 0x000000);

				// 各点を移動してソートする
				for each(var point:MapPoint in points){
					point.pt = matrix.transformPoint(point.pt);
				}
				points.sort(function(a:MapPoint, b:MapPoint):int{
					return (b.pt.y < a.pt.y ? 1 : b.pt.y > a.pt.y ? -1 : 0);
				});

				// 描画する
				for each (point in points){
					pt = point.pt.clone();
					pt.x *= F / (F - pt.y) * .6;
					pt.y *= F / (F - pt.y) * .3;
					pt.x += WIDTH * .5;
					pt.y = pt.y + hh * .5;

					var h1:int = point.z;
					if (h1 != 0){
						var h2:int = h1 * ELEVATION / 5000;
						rt.top = rt.bottom - h2 - 1;
						pt.y -= h2;
						bmd.copyPixels(bmdLine, rt, pt);
					} else {
						bmd.setPixel(pt.x, pt.y, COLOR_SEA);
					}
				}

				bmdCache[counter] = bmd;
			}

			bmp.bitmapData = bmdCache[counter];
			counter = (counter + 1) % FRAME;
		});
	}

	private function createBmdLine():BitmapData{
		var bmdLine:BitmapData = new BitmapData(1, ELEVATION * 2);
		var lineSprite:Sprite = new Sprite();
		var matrix:Matrix = new Matrix();
		matrix.createGradientBox(1, ELEVATION, -Math.PI / 2, 0, ELEVATION);
		lineSprite.graphics.lineStyle(10, 0, 1, false, "normal", "none");
		lineSprite.graphics.lineGradientStyle("linear", COLORS, [1, 1, 1, 1, 1, 1], [0, 0x33, 0x66, 0x99, 0xcc, 0xff], matrix);
		lineSprite.graphics.lineTo(0, ELEVATION * 2);
		bmdLine.draw(lineSprite);
		return bmdLine;
	}
}
}

import flash.geom.Point;
import flash.geom.Matrix;
import flash.display.Sprite;

var COLOR_SEA:uint = 0x7ebeea;
var COLORS:Array = [0x149614, 0xc8c800, 0x874700, 0x532f06, 0x675400, 0xffffff];
var HEIGHT:Number = 30;

class MapPoint{
	public var pt:Point;
	public var z:uint;
	public var color:uint;

	public function MapPoint(x:int, y:int, z:int){
		pt = new Point(x, y);
		this.z = z;
	}
}

const E100N40_DATA:String = "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"
