/**
 * Copyright widged ( http://wonderfl.net/user/widged )
 * MIT License ( http://www.opensource.org/licenses/mit-license.php )
 * Downloaded from: http://wonderfl.net/c/297d
 */

/**
 * Pseudo-random Color Scheme Generator  
 * 
 * Ever wondered how designers create these wonderful color schemes. A typical strategy is to select pure hues,  
 * copy the line of color swatches multiple times, then adding an overlay of some level of black for accents or 
 * an overlay of some level of white for highlights. 
 * 
 * This is explained in some details in articles like: 
 * How to Get a Professional Look With Color
 * http://www.webdesignerdepot.com/2009/12/how-to-get-a-professional-look-with-color/
 * 
 * Here is an attempt to produce something such schemes pseudo randomly. First, a list of hues is generated that 
 * contains the base color. Then, saturation or brightness is added. 
 * 
 * Click on any swatch to redraw the full palette, with new pseudo-random values. 
 * The number of swatches can be changed in the call createDistinctList(9). Replace 9 with another value. 
 */
 package
{
    import com.bit101.components.*;
    
    import flash.display.Shape;
    import flash.display.Sprite;
    import flash.events.MouseEvent;
    import flash.geom.Matrix;
    import flash.text.TextField;
    import flash.text.TextFormat;
    
    [SWF(width = "465", height = "465", backgroundColor='#FFFFFF')]
    public class ColorPaletteGenerator extends Sprite
    {
        private var palette:Sprite;
        public function ColorPaletteGenerator()
        {
            drawRandomPalette();
        }
        
        private function onClick(event:MouseEvent):void
        {
            this.removeChild(palette);
            drawRandomPalette();
        }
        
        private function drawRandomPalette():void
        {
            palette = new Sprite();
            palette.x = 20;
            palette.y = 20;
            var hueList:Array = createDistinctList(9);
            // trace(hueList.length, "--", hueList);
            
            var posX:int = 0, posY:int = 0;
            for (var i:int = 0; i < hueList.length; i++)
            {
                var hue:Number = hueList[i];
                var swatch:Sprite = drawTintGradation(hue);
                swatch.x = posX; swatch.y = posY;
                posX += 140; if(i && (i+1) % 3 == 0) { posX = 0; posY +=80;}
                palette.addChild(swatch);
            }
            this.addChild(palette);
        }
        
        private function createDistinctList(qty:int):Array
        {
            var list:Array = [];
            var val:Number = 0, inc:Number;
            // first try and generate a list of numbers between 0 and 1 that are not too similar 
            // that is that are characterized by a minimum increment between them.
            // Generate a list that is barely bigger than the final list, with values as 
            // distinct as possible. 
            while (list.length < (qty * 3) && val <= 1) 
            {
                inc = NumberUtils.roundDec((Math.random() + 0.001) / (qty*1), 3);
                if(inc < (1 / (qty*2))) continue;
                val += inc;
                if(val > 1) continue;
                list.push(NumberUtils.roundDec(val,3));     
            }
            // trace(list.length);
            // pick randomly in the master list, up until the quantity specified 
            var list2:Array = [];
            while (list2.length < qty)
            {
                var idx:int = NumberUtils.random(list.length);
                list2.push(list.splice(idx,1)); 
            }
            list2.sort();
            return list2;
        }

        private function drawTintGradation(hue:Number):Sprite
        {
            var sprite:Sprite = new Sprite();
            
            // swatch part
            // will react to a mouseclick
            var swatch:Sprite = new Sprite();
            var rgbList:Array = [];
            for(var i:int=1; i<=3; i++){
                var color:uint;
                if(i == 3) {
                    // tint with black
                    color = ColorUtils.hsbToRgb(hue,1,80/100); 
                } else if(i == 2) {
                    // tint with white
                    color = ColorUtils.hsbToRgb(hue,35/100,1); 
                } else if(i == 1) {
                    // tint with white
                    color = ColorUtils.hsbToRgb(hue,15/100,1); 
                } 
                var rgb:Object = ColorUtils.hexToRGB (color);
                rgbList.push(rgb.r + ", " + rgb.g + ", " + rgb.b);
                
                var shape:Shape = new Shape();
                var size:Number = 60 - i*10;
                shape.graphics.beginFill(color,0.8);
                shape.graphics.drawRect(0,0,size, size);
                swatch.addChild(shape);
                swatch.addEventListener(MouseEvent.CLICK, onClick, false, 0, true);
                sprite.addChild(swatch);

            }
            
            // text part
            // It should be possible to click to capture the text without messing
            // up with the palette
            var rgbText:Text = new Text(sprite, 50, 0, rgbList.join("\n"));
            rgbText.width = 75;
            rgbText.height = 50;
            sprite.addChild(rgbText);

            return sprite;
        }

    }
    
}

class ColorUtils
{
    
    public static function hexToRGB ( hex:uint ):Object 
    {
        var obj:Object = {r:0, g:0, b:0}; 
        obj.r = hex >> 16;
        var temp:uint = hex ^ obj.r << 16;
        obj.g = temp >> 8;
        obj.b = temp ^ obj.g << 8;
        
        return obj;
    }

    /** HSB 2 RGB
     * @param hue  (0 - 360)
     * @param saturation (0.0 to 1.0)
     * @param brightness (0.0 to 1.0)
     * @return RGB
     */
    public static function hsbToRgb(hue:Number, saturation:Number, brightness:Number):uint {
        var r:int = 0;
        var g:int = 0;
        var b:int = 0;
        if (saturation == 0) {
            r = g = b = brightness * 255.0 + 0.5;
        } else {
            var h:Number = (hue - Math.floor(hue)) * 6.0;
            var f:Number = h - Math.floor(h);
            var p:Number = brightness * (1.0 - saturation);
            var q:Number = brightness * (1.0 - saturation * f);
            var t:Number = brightness * (1.0 - (saturation * (1.0 - f)));
            switch (int(h)) {
                case 0:
                    r = brightness * 255.0 + 0.5;
                    g = t * 255.0 + 0.5;
                    b = p * 255.0 + 0.5;
                    break;
                case 1:
                    r = q * 255.0 + 0.5;
                    g = brightness * 255.0 + 0.5;
                    b = p * 255.0 + 0.5;
                    break;
                case 2:
                    r = p * 255.0 + 0.5;
                    g = brightness * 255.0 + 0.5;
                    b = t * 255.0 + 0.5;
                    break;
                case 3:
                    r = p * 255.0 + 0.5;
                    g = q * 255.0 + 0.5;
                    b = brightness * 255.0 + 0.5;
                    break;
                case 4:
                    r = t * 255.0 + 0.5;
                    g = p * 255.0 + 0.5;
                    b = brightness * 255.0 + 0.5;
                    break;
                case 5:
                    r = brightness * 255.0 + 0.5;
                    g = p * 255.0 + 0.5;
                    b = q * 255.0 + 0.5;
                    break;
            }
        }
        return (r << 16) | (g << 8) | (b << 0);
    }            
}

class NumberUtils
{
    public static function random(max:int):int
    {
        return Math.round(Math.random() * max);
    }
    
    public static function roundDec(value:Number, decQty:int):Number
    {
        if(decQty == 0) return Math.round(value);
        if(value ==0) return 0;
        var pow:int = Math.pow(10, decQty);
        return Math.round(value * pow) / pow        
    }

}